<?php
/**
 * Plugin Name: Viewers.com
 * Plugin URI: https://viewers.com
 * Description: Embed the Viewers.com widget on your WordPress site.
 * Version: 1.2.0
 * Author: Viewers.com
 * Author URI: https://viewers.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: viewers-com
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class ViewersCom {
    
    private static $instance = null;
    private $option_name = 'viewers_com_settings';
    private $cron_hook = 'viewers_com_check_embed';
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        // Admin hooks
        add_action('admin_menu', array($this, 'add_settings_page'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        
        // AJAX handlers
        add_action('wp_ajax_viewers_com_save_api_key', array($this, 'ajax_save_api_key'));
        add_action('wp_ajax_viewers_com_clear_cache', array($this, 'ajax_clear_cache'));
        
        // Cron hook
        add_action($this->cron_hook, array($this, 'fetch_embed_code'));
        
        // Frontend hook - inject script before </body>
        add_action('wp_footer', array($this, 'inject_embed_code'), 999);
        
        // Activation/Deactivation hooks
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }
    
    /**
     * Plugin activation
     */
    public function activate() {
        // Schedule cron job every 5 minutes
        if (!wp_next_scheduled($this->cron_hook)) {
            wp_schedule_event(time(), 'five_minutes', $this->cron_hook);
        }
    }
    
    /**
     * Plugin deactivation
     */
    public function deactivate() {
        // Clear scheduled cron
        wp_clear_scheduled_hook($this->cron_hook);
    }
    
    /**
     * Add custom cron schedule for 5 minutes
     */
    public static function add_cron_interval($schedules) {
        $schedules['five_minutes'] = array(
            'interval' => 300,
            'display'  => __('Every 5 Minutes', 'viewers-com'),
        );
        return $schedules;
    }
    
    /**
     * Add settings page to admin menu
     */
    public function add_settings_page() {
        add_options_page(
            __('Viewers.com Embed', 'viewers-com'),
            __('Viewers.com Embed', 'viewers-com'),
            'manage_options',
            'viewers-com-embed',
            array($this, 'render_settings_page')
        );
    }
    
    /**
     * Register plugin settings
     */
    public function register_settings() {
        register_setting($this->option_name, $this->option_name, array(
            'sanitize_callback' => array($this, 'sanitize_settings'),
        ));
    }
    
    /**
     * Sanitize settings
     */
    public function sanitize_settings($input) {
        $sanitized = array();
        if (isset($input['api_key'])) {
            $sanitized['api_key'] = sanitize_text_field($input['api_key']);
        }
        if (isset($input['embed_code'])) {
            $sanitized['embed_code'] = $input['embed_code'];
        }
        if (isset($input['last_updated'])) {
            $sanitized['last_updated'] = intval($input['last_updated']);
        }
        if (isset($input['connected'])) {
            $sanitized['connected'] = (bool) $input['connected'];
        }
        return $sanitized;
    }
    
    /**
     * Enqueue admin scripts
     */
    public function enqueue_admin_scripts($hook) {
        if ('settings_page_viewers-com-embed' !== $hook) {
            return;
        }
        
        wp_enqueue_style(
            'viewers-com-admin',
            plugin_dir_url(__FILE__) . 'css/admin.css',
            array(),
            '1.0.0'
        );
        
        wp_enqueue_script(
            'viewers-com-admin',
            plugin_dir_url(__FILE__) . 'js/admin.js',
            array('jquery'),
            '1.0.0',
            true
        );
        
        wp_localize_script('viewers-com-admin', 'viewersComAdmin', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('viewers_com_nonce'),
        ));
    }
    
    /**
     * Get the API endpoint URL
     */
    private function get_api_endpoint($api_key) {
        $base_url = 'https://app.viewers.com/api-wordpress-embed.php';
        
        return add_query_arg('api_key', $api_key, $base_url);
    }
    
    /**
     * Fetch embed code from API
     */
    public function fetch_embed_code($api_key = null) {
        $settings = get_option($this->option_name, array());
        
        if (null === $api_key) {
            $api_key = isset($settings['api_key']) ? $settings['api_key'] : '';
        }
        
        if (empty($api_key)) {
            return false;
        }
        
        $endpoint = $this->get_api_endpoint($api_key);
        
        $response = wp_remote_get($endpoint, array(
            'timeout' => 30,
            'sslverify' => false, // For local development
        ));
        
        if (is_wp_error($response)) {
            return false;
        }
        
        $body = wp_remote_retrieve_body($response);
        $status_code = wp_remote_retrieve_response_code($response);
        
        if ($status_code !== 200 || empty($body)) {
            // Mark as disconnected
            $settings['connected'] = false;
            $settings['embed_code'] = '';
            update_option($this->option_name, $settings);
            return false;
        }
        
        // Clean up the embed code - remove surrounding quotes and unescape
        $embed_code = $this->clean_embed_code($body);
        
        // Check if API returned false or empty/invalid response
        if ($embed_code === 'false' || $embed_code === '' || strpos($embed_code, '<script') === false) {
            // Mark as disconnected - site may have been deleted
            $settings['connected'] = false;
            $settings['embed_code'] = '';
            update_option($this->option_name, $settings);
            return false;
        }
        
        // Update settings
        $settings['embed_code'] = $embed_code;
        $settings['last_updated'] = time();
        $settings['connected'] = true;
        
        update_option($this->option_name, $settings);
        
        return $embed_code;
    }
    
    /**
     * Clean up the embed code from API response
     */
    private function clean_embed_code($code) {
        // Remove surrounding quotes if present
        $code = trim($code, '"\'');
        
        // Unescape backslashes
        $code = stripslashes($code);
        
        return $code;
    }
    
    /**
     * AJAX handler for saving API key
     */
    public function ajax_save_api_key() {
        check_ajax_referer('viewers_com_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Unauthorized', 'viewers-com')));
        }
        
        $api_key = isset($_POST['api_key']) ? sanitize_text_field($_POST['api_key']) : '';
        
        if (empty($api_key)) {
            wp_send_json_error(array('message' => __('API key is required', 'viewers-com')));
        }
        
        // Save the API key
        $settings = get_option($this->option_name, array());
        $settings['api_key'] = $api_key;
        update_option($this->option_name, $settings);
        
        // Fetch the embed code
        $embed_code = $this->fetch_embed_code($api_key);
        
        if ($embed_code) {
            wp_send_json_success(array(
                'message' => __('Connected successfully!', 'viewers-com'),
                'embed_code' => $embed_code,
                'last_updated' => date('Y-m-d H:i:s'),
            ));
        } else {
            // Still save the key but mark as not connected
            $settings['connected'] = false;
            update_option($this->option_name, $settings);
            
            wp_send_json_error(array('message' => __('Could not connect to Viewers.com API. Please check your API key.', 'viewers-com')));
        }
    }
    
    /**
     * AJAX handler for clearing cache and rechecking
     */
    public function ajax_clear_cache() {
        check_ajax_referer('viewers_com_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Unauthorized', 'viewers-com')));
        }
        
        $embed_code = $this->fetch_embed_code();
        
        if ($embed_code) {
            wp_send_json_success(array(
                'message' => __('Cache cleared and embed code updated!', 'viewers-com'),
                'embed_code' => $embed_code,
                'last_updated' => date('Y-m-d H:i:s'),
            ));
        } else {
            wp_send_json_error(array('message' => __('Could not fetch embed code. Please check your API key.', 'viewers-com')));
        }
    }
    
    /**
     * Inject embed code into footer
     */
    public function inject_embed_code() {
        // Don't inject in admin
        if (is_admin()) {
            return;
        }
        
        $settings = get_option($this->option_name, array());
        
        // Don't inject if not connected or no valid embed code
        if (empty($settings['embed_code']) || empty($settings['connected'])) {
            return;
        }
        
        // Extra safety check - only inject if it looks like a valid script tag
        if (strpos($settings['embed_code'], '<script') === false) {
            return;
        }
        
        // Output the embed code (it's already a script tag)
        echo "\n" . $settings['embed_code'] . "\n";
    }
    
    /**
     * Render settings page
     */
    public function render_settings_page() {
        $settings = get_option($this->option_name, array());
        $api_key = isset($settings['api_key']) ? $settings['api_key'] : '';
        $embed_code = isset($settings['embed_code']) ? $settings['embed_code'] : '';
        $connected = isset($settings['connected']) ? $settings['connected'] : false;
        $last_updated = isset($settings['last_updated']) ? $settings['last_updated'] : 0;
        ?>
        <div class="wrap viewers-com-wrap">
            <h1><?php echo esc_html__('Viewers.com Embed', 'viewers-com'); ?></h1>
            
            <div class="viewers-com-container">
                <div class="viewers-com-main">
                    <div class="viewers-com-card">
                        <h2><?php echo esc_html__('API Configuration', 'viewers-com'); ?></h2>
                        
                        <div class="viewers-com-form-group">
                            <label for="viewers-api-key"><?php echo esc_html__('API Key', 'viewers-com'); ?></label>
                            <input type="text" id="viewers-api-key" class="regular-text" value="<?php echo esc_attr($api_key); ?>" placeholder="<?php echo esc_attr__('Enter your Viewers.com API key', 'viewers-com'); ?>">
                            <button type="button" id="viewers-save-key" class="button button-primary">
                                <?php echo esc_html__('Save & Connect', 'viewers-com'); ?>
                            </button>
                        </div>
                        
                        <div id="viewers-status" class="viewers-com-status <?php echo $connected ? 'connected' : ''; ?>">
                            <?php if ($connected) : ?>
                                <div class="viewers-com-notice success">
                                    <span class="dashicons dashicons-yes-alt"></span>
                                    <div>
                                        <strong><?php echo esc_html__('Connected to Viewers.com', 'viewers-com'); ?></strong>
                                        <p class="viewers-com-note"><?php echo esc_html__('Note: It can take a few minutes for new changes to start displaying on your site.', 'viewers-com'); ?></p>
                                    </div>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <?php if ($connected && !empty($embed_code)) : ?>
                    <div class="viewers-com-card">
                        <h2><?php echo esc_html__('Embed Status', 'viewers-com'); ?></h2>
                        
                        <div class="viewers-com-info">
                            <p>
                                <strong><?php echo esc_html__('Last Updated:', 'viewers-com'); ?></strong>
                                <span id="viewers-last-updated"><?php echo $last_updated ? date('Y-m-d H:i:s', $last_updated) : __('Never', 'viewers-com'); ?></span>
                            </p>
                            <p class="description"><?php echo esc_html__('The embed code is automatically checked every 5 minutes for updates.', 'viewers-com'); ?></p>
                        </div>
                        
                        <div class="viewers-com-actions">
                            <button type="button" id="viewers-clear-cache" class="button button-secondary">
                                <span class="dashicons dashicons-update"></span>
                                <?php echo esc_html__('Clear Cache & Recheck', 'viewers-com'); ?>
                            </button>
                        </div>
                        
                        <div class="viewers-com-code-preview">
                            <h3><?php echo esc_html__('Current Embed Code', 'viewers-com'); ?></h3>
                            <pre id="viewers-embed-preview"><?php echo esc_html($embed_code); ?></pre>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
                
                <?php if ($connected && !empty($embed_code)) : ?>
                <div class="viewers-com-sidebar">
                    <div class="viewers-com-card">
                        <h2><?php echo esc_html__('Configure Widget', 'viewers-com'); ?></h2>
                        
                        <div class="viewers-com-info">
                            <p><?php echo esc_html__('To customize your widget appearance, behavior, and settings, visit your Viewers.com dashboard. Changes made there will automatically sync to your site.', 'viewers-com'); ?></p>
                        </div>
                        
                        <div class="viewers-com-actions">
                            <a href="https://app.viewers.com/configure.php" target="_blank" rel="noopener noreferrer" class="button button-primary">
                                <span class="dashicons dashicons-admin-generic"></span>
                                <?php echo esc_html__('Change Configuration', 'viewers-com'); ?>
                            </a>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
            </div>
            
            <?php if ($connected && !empty($embed_code)) : ?>
            <!-- Load the widget on this admin page for preview -->
            <?php echo $embed_code; ?>
            <?php endif; ?>
            
            <div id="viewers-loading" class="viewers-com-loading" style="display: none;">
                <span class="spinner is-active"></span>
                <span><?php echo esc_html__('Processing...', 'viewers-com'); ?></span>
            </div>
        </div>
        <?php
    }
}

// Add custom cron interval
add_filter('cron_schedules', array('ViewersCom', 'add_cron_interval'));

// Initialize plugin
ViewersCom::get_instance();

